/*
 * USB4all driver - 1.0
 *
 * Copyright (C) 2007 Andres Aguirre
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License as
 *      published by the Free Software Foundation, version 2.
 *
 */


//#include <linux/config.h> para versiones de kernels viejos
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/kref.h>
#include <asm/uaccess.h>
#include <linux/usb.h>
#include <linux/usb/ch9.h>

/* Define these values to match your devices */
#define USB4all_VENDOR_ID	 0x04d8
#define USB4all_PRODUCT_ID	 0x000c

/* Params for SET_TRANSFER_TYPE IOCTL */
#define CTRL	0
#define INT	1
#define BULK	2
#define ISO	3

/* IOCTL commands */
#define GET_DEVICE_DESCRIPTOR	 	0x0001
#define SET_DESC_INDEX		 	0x0002
#define SET_IN_ENDPOINT		 	0x0003
#define SET_OUT_ENDPOINT   	 	0x0004
#define GET_ENDPOINT_DESCRIPTOR  	0x0005
#define GET_INTERFACE_DESCRIPTOR 	0x0006
#define GET_CONFIGURATION_DESCRIPTOR 	0x0007
#define SET_TRANSFER_TYPE	 	0x0008
#define SET_TIMEOUT			0x0009
#define GET_STRING_DESCRIPTOR		0x000a
#define DEBUG_STRING_DESCRIPTOR		0x000b

/* table of devices that work with this driver */
static struct usb_device_id USB4all_table [] = {
	{ USB_DEVICE(USB4all_VENDOR_ID, USB4all_PRODUCT_ID) },
	{ }					/* Terminating entry */
};
MODULE_DEVICE_TABLE (usb, USB4all_table);


/* Get a minor range for your devices from the usb maintainer */
#define USB4all_MINOR_BASE	192

/* our private defines. if this grows any larger, use your own .h file */
#define MAX_TRANSFER		( PAGE_SIZE - 512 )
#define WRITES_IN_FLIGHT	8

/* Structure to hold all of our device specific stuff */
struct usb_U4all {
	struct usb_device *	udev;			/* the usb device for this device */
	struct usb_interface *	interface;		/* the interface for this device */
	struct semaphore	limit_sem;		/* limiting the number of writes in progress */
	unsigned char *		in_buffer;		/* the buffer to receive data */
	size_t			in_size;		/* the size of the receive buffer */
	__u8			in_endpointAddr;	/* the address of the bulk in endpoint */
	__u8			out_endpointAddr;	/* the address of the bulk out endpoint */
	struct kref		kref;
};
#define to_U4all_dev(d) container_of(d, struct usb_U4all, kref)

static struct usb_driver U4all_driver;

struct endpointSizeItem {
	u_int16_t wMaxPacketSize;
	u_int8_t  bEndpointAddress;
};
//indice del descriptor pedido mediante ioctl
unsigned char descindex = 0;

struct endpointSizeItem endpointSizes[15];

static int timeout = 10000;
static int transferType = BULK;

static void unicode_to_ascii            (char *string, __le16 *unicode, int unicode_size);

static void U4all_delete(struct kref *kref)
{	
	struct usb_U4all *dev = to_U4all_dev(kref);

	usb_put_dev(dev->udev);
	kfree (dev->in_buffer);
	kfree (dev);
}

static int U4all_open(struct inode *inode, struct file *file)
{
	struct usb_U4all *dev;
	struct usb_interface *interface;
	int subminor;
	int retval = 0;

	subminor = iminor(inode);

	interface = usb_find_interface(&U4all_driver, subminor);
	if (!interface) {
		err ("%s - error, can't find device for minor %d",
		     __FUNCTION__, subminor);
		retval = -ENODEV;
		goto exit;
	}

	dev = usb_get_intfdata(interface);
	if (!dev) {
		retval = -ENODEV;
		goto exit;
	}

	/* increment our usage count for the device */
	kref_get(&dev->kref);

	/* save our object in the file's private structure */
	file->private_data = dev;

exit:
	return retval;
}

static int U4all_release(struct inode *inode, struct file *file)
{
	struct usb_U4all *dev;

	dev = (struct usb_U4all *)file->private_data;
	if (dev == NULL)
		return -ENODEV;

	/* decrement the count on our device */
	kref_put(&dev->kref, U4all_delete);
	return 0;
}

static ssize_t U4all_read(struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	struct usb_U4all *dev;
	int retval = 0;
	int bytes_read;

	dev = (struct usb_U4all *)file->private_data;
	
	/* do a blocking bulk read to get data from the device */
	retval = usb_bulk_msg(dev->udev,
			      usb_rcvbulkpipe(dev->udev, dev->in_endpointAddr),
			      dev->in_buffer,
			      min(dev->in_size, count),
			      &bytes_read, 10000);

	/* if the read was successful, copy the data to userspace */
	if (!retval) {
		if (copy_to_user(buffer, dev->in_buffer, bytes_read))
			retval = -EFAULT;
		else
			retval = bytes_read;
	}

	return retval;
}

static void U4all_write_bulk_callback(struct urb *urb, struct pt_regs *regs)
{
	struct usb_U4all *dev;

	dev = (struct usb_U4all *)urb->context;

	/* sync/async unlink faults aren't errors */
	if (urb->status && 
	    !(urb->status == -ENOENT || 
	      urb->status == -ECONNRESET ||
	      urb->status == -ESHUTDOWN)) {
		err("%s - nonzero write bulk status received: %d",
		    __FUNCTION__, urb->status);
	}

	/* free up our allocated buffer */
	usb_buffer_free(urb->dev, urb->transfer_buffer_length, 
			urb->transfer_buffer, urb->transfer_dma);
	up(&dev->limit_sem);
}

static ssize_t U4all_write(struct file *file, const char *user_buffer, size_t count, loff_t *ppos)
{
	struct usb_U4all *dev;
	int retval = 0;
	struct urb *urb = NULL;
	char *buf = NULL;
	size_t writesize = min(count, (size_t)MAX_TRANSFER);

	dev = (struct usb_U4all *)file->private_data;

	/* verify that we actually have some data to write */
	if (count == 0)
		goto exit;

	/* limit the number of URBs in flight to stop a user from using up all RAM */
	if (down_interruptible(&dev->limit_sem)) {
		retval = -ERESTARTSYS;
		goto exit;
	}

	/* create a urb, and a buffer for it, and copy the data to the urb */
	urb = usb_alloc_urb(0, GFP_KERNEL);
	if (!urb) {
		retval = -ENOMEM;
		goto error;
	}

	buf = usb_buffer_alloc(dev->udev, writesize, GFP_KERNEL, &urb->transfer_dma);
	if (!buf) {
		retval = -ENOMEM;
		goto error;
	}

	if (copy_from_user(buf, user_buffer, writesize)) {
		retval = -EFAULT;
		goto error;
	}

	/* initialize the urb properly */
	usb_fill_bulk_urb(urb, dev->udev,
			  usb_sndbulkpipe(dev->udev, dev->out_endpointAddr),
			  buf, writesize, (usb_complete_t)U4all_write_bulk_callback, dev);
	urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;

	/* send the data out the bulk port */
	retval = usb_submit_urb(urb, GFP_KERNEL);
	if (retval) {
		err("%s - failed submitting write urb, error %d", __FUNCTION__, retval);
		goto error;
	}

	/* release our reference to this urb, the USB core will eventually free it entirely */
	usb_free_urb(urb);

exit:
	return writesize;

error:
	usb_buffer_free(dev->udev, writesize, buf, urb->transfer_dma);
	usb_free_urb(urb);
	up(&dev->limit_sem);
	return retval;
}

static int usb4all_ioctl(struct inode *inode, struct file *file, unsigned int code, unsigned long user_data){
        struct usb_U4all *u4alldev;
	struct usb_device* dev;
	int ret;
	struct usb_device_descriptor* descriptor; 
        u4alldev = (struct usb_U4all *)file->private_data;
        if (u4alldev == NULL)
                return -ENODEV;
	dev = u4alldev->udev;
	switch (code) {
	case SET_DESC_INDEX: {
		ret = get_user(descindex, (unsigned char*)user_data);
		return ret;
		break;
	}
        case SET_TIMEOUT: {
                ret = get_user(timeout, (unsigned char*)user_data);
                return ret;
                break;
        }
	case SET_TRANSFER_TYPE: {
                ret = get_user(transferType, (const char*)user_data);
                return ret;
                break;
        }
	case SET_IN_ENDPOINT: {
		ret = get_user(u4alldev->in_endpointAddr, (unsigned char*)user_data);
		u4alldev->in_size = endpointSizes[descindex].wMaxPacketSize;
		return ret;
		break;
	}
	case SET_OUT_ENDPOINT: {
		ret = get_user(u4alldev->out_endpointAddr, (unsigned char*)user_data);
		return ret;
		break;
	}
	case GET_STRING_DESCRIPTOR: {
	        struct usb_string_descriptor StringDesc;
        	struct usb_string_descriptor *pStringDesc;
        	dbg("%s - USB String ID = %d", __FUNCTION__, Id );

        	if (!usb_get_descriptor(dev, USB_DT_STRING, descindex, &StringDesc, sizeof(StringDesc))) {
               		return 0;
        	}

        	pStringDesc = kmalloc (StringDesc.bLength, GFP_KERNEL);

        	if (!pStringDesc) {
                	return 0;
        	}

        	if (!usb_get_descriptor(dev, USB_DT_STRING, descindex, pStringDesc, StringDesc.bLength )) {
                	kfree(pStringDesc);
                	return 0;
        	}

        	//unicode_to_ascii(string,  pStringDesc->wData,     pStringDesc->bLength/2-1);
		ret = copy_to_user((unsigned int __user *) user_data, pStringDesc, sizeof(*pStringDesc));
		//ret = copy_to_user((unsigned int __user *) user_data, string, strlen(string)+1);
        	kfree(pStringDesc);
        	return ret;
		break;
	}
	case DEBUG_STRING_DESCRIPTOR: {
	        struct usb_string_descriptor StringDesc;
                struct usb_string_descriptor *pStringDesc;
                char* string;
                dbg("%s - USB String ID = %d", __FUNCTION__, Id );

                if (!usb_get_descriptor(dev, USB_DT_STRING, descindex, &StringDesc, sizeof(StringDesc))) {
                        return 0;
                }

                pStringDesc = kmalloc (StringDesc.bLength, GFP_KERNEL);

                if (!pStringDesc) {
                        return 0;
                }

                if (!usb_get_descriptor(dev, USB_DT_STRING, descindex, pStringDesc, StringDesc.bLength )) {
                        kfree(pStringDesc);
                        return 0;
                }
		string = kmalloc (StringDesc.bLength, GFP_KERNEL);
		unicode_to_ascii(string,  pStringDesc->wData,     pStringDesc->bLength/2-1);
		ret = copy_to_user((unsigned int __user *) user_data, string, strlen(string)+1);
                kfree(pStringDesc);
                return ret;
                break;
	}	
	case GET_DEVICE_DESCRIPTOR: {
 		descriptor = kmalloc(sizeof *descriptor, GFP_NOIO);
		ret = usb_get_descriptor(dev, USB_DT_DEVICE, descindex, descriptor, USB_DT_DEVICE_SIZE);
		if (ret < 0) {
		    kfree(descriptor);
		    return ret;
	        }
		ret = copy_to_user((unsigned int __user *) user_data, descriptor, sizeof(*descriptor)); 
		return ret;
		break;
	}
	case GET_ENDPOINT_DESCRIPTOR: {
                struct usb_interface *interface;
                struct usb_host_interface *iface_desc;
                struct usb_endpoint_descriptor *endpoint;
                struct endpointSizeItem item;
		int subminor;
                subminor = iminor(inode);
                interface = usb_find_interface(&U4all_driver, subminor);
                endpoint = kmalloc(USB_DT_ENDPOINT_SIZE, GFP_NOIO);
		if (!interface) {
                        dbg ("%s - error, can't find device for minor %d",
                        __FUNCTION__, subminor);
                        ret = -ENODEV;
                        return ret;
                }
                iface_desc = interface->cur_altsetting;
		endpoint = &iface_desc->endpoint[descindex].desc;
              	item.bEndpointAddress = endpoint->bEndpointAddress;
		item.wMaxPacketSize = endpoint->wMaxPacketSize; 
		endpointSizes[descindex] = item;
		ret = copy_to_user((unsigned int __user *) user_data, endpoint, USB_DT_ENDPOINT_SIZE);
                return ret;
                break;
	}
	case GET_INTERFACE_DESCRIPTOR: {
		struct usb_interface *interface;
		struct usb_host_interface *iface_desc;
        	int subminor;
		subminor = iminor(inode);
		iface_desc = kmalloc(USB_DT_INTERFACE_SIZE, GFP_NOIO);
		interface = usb_find_interface(&U4all_driver, subminor);
        	if (!interface) {
                	dbg ("%s - error, can't find device for minor %d",
                   	__FUNCTION__, subminor);
                	ret = -ENODEV;
                	return ret;
        	}
		iface_desc = interface->cur_altsetting;
                ret = copy_to_user((unsigned int __user *) user_data, iface_desc, USB_DT_INTERFACE_SIZE);
                return ret;
                break;
        }
	case GET_CONFIGURATION_DESCRIPTOR: {
		struct usb_config_descriptor* config;
		config = kmalloc(USB_DT_CONFIG_SIZE, GFP_NOIO);
		//unsigned char bufereo[255];
		//ret =  usb_get_descriptor(dev, USB_DT_CONFIG, descindex, bufereo, 256);
		ret =  usb_get_descriptor(dev, USB_DT_CONFIG, descindex, config, USB_DT_CONFIG_SIZE);
	        if (ret < 0) {
                    return ret;
                }
                ret = copy_to_user((unsigned int __user *) user_data, config, USB_DT_CONFIG_SIZE);
                return ret;
                break;
	}
	default:
		return -ENOSYS;
	}
}

static struct file_operations U4all_fops = {
        .owner =        THIS_MODULE,
        .read =         U4all_read,
        .write =        U4all_write,
        .open =         U4all_open,
	.release =      U4all_release,
        .ioctl =        usb4all_ioctl,
};




/*
 *  *  * usb class driver info in order to get a minor number from the usb core,
 *   *   * and to have the device registered with the driver core
 *    *    */
static struct usb_class_driver U4all_class = {
        .name =         "usb4all%d",
        .fops =         &U4all_fops,
        .minor_base =   USB4all_MINOR_BASE,
};


static int U4all_probe(struct usb_interface *interface, const struct usb_device_id *id)
{
	struct usb_U4all *dev = NULL;
	//size_t buffer_size;
	//int i;
	int retval = -ENOMEM;

	/* allocate memory for our device state and initialize it */
	dev = kzalloc(sizeof(*dev), GFP_KERNEL);
	if (dev == NULL) {
		err("Out of memory");
		goto error;
	}
	kref_init(&dev->kref);
	sema_init(&dev->limit_sem, WRITES_IN_FLIGHT);

	dev->udev = usb_get_dev(interface_to_usbdev(interface));
	dev->interface = interface;

	dev->in_size = 0x0040;
	/* set up the endpoint information */
	/* use only the first bulk-in and bulk-out endpoints */
	/*iface_desc = interface->cur_altsetting;
	for (i = 0; i < iface_desc->desc.bNumEndpoints; ++i) {
		endpoint = &iface_desc->endpoint[i].desc;

		if (!dev->bulk_in_endpointAddr &&
		    ((endpoint->bEndpointAddress & USB_ENDPOINT_DIR_MASK)
					== USB_DIR_IN) &&
		    ((endpoint->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK)
					== USB_ENDPOINT_XFER_BULK)) {*/
			/* we found a bulk in endpoint */
	/*		buffer_size = le16_to_cpu(endpoint->wMaxPacketSize);
			dev->bulk_in_size = buffer_size;
			dev->bulk_in_endpointAddr = endpoint->bEndpointAddress;*/
			dev->in_buffer = kmalloc(0x0040/*buffer_size*/, GFP_KERNEL);
			if (!dev->in_buffer) {
				err("Could not allocate bulk_in_buffer");
				goto error;
			}/*
		}

		if (!dev->bulk_out_endpointAddr &&
		    ((endpoint->bEndpointAddress & USB_ENDPOINT_DIR_MASK)
					== USB_DIR_OUT) &&
		    ((endpoint->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK)
					== USB_ENDPOINT_XFER_BULK)) {*/
			/* we found a bulk out endpoint */
	/*		dev->bulk_out_endpointAddr = endpoint->bEndpointAddress;
		}
	}
	if (!(dev->bulk_in_endpointAddr && dev->bulk_out_endpointAddr)) {
		err("Could not find both bulk-in and bulk-out endpoints");
		goto error;
	}*/

	/* save our data pointer in this interface device */
	usb_set_intfdata(interface, dev);

	/* we can register the device now, as it is ready */
	retval = usb_register_dev(interface, &U4all_class);
	if (retval) {
		/* something prevented us from registering this driver */
		err("Not able to get a minor for this device.");
		usb_set_intfdata(interface, NULL);
		goto error;
	}

	/* let the user know what node this device is now attached to */
	info("USB4all device now attached to usb4all-%d", interface->minor);
	return 0;

error:
	if (dev)
		kref_put(&dev->kref, U4all_delete);
	return retval;
}

static void U4all_disconnect(struct usb_interface *interface)
{
	struct usb_U4all *dev;
	int minor = interface->minor;

	/* prevent U4all_open() from racing U4all_disconnect() */
	lock_kernel();

	dev = usb_get_intfdata(interface);
	usb_set_intfdata(interface, NULL);

	/* give back our minor */
	usb_deregister_dev(interface, &U4all_class);

	unlock_kernel();

	/* decrement our usage count */
	kref_put(&dev->kref, U4all_delete);

	info("USB4all #%d now disconnected", minor);
}


static struct usb_driver U4all_driver = {
        .name =         "USB4all baseboard",
        .probe =        U4all_probe,
        .disconnect =   U4all_disconnect,
        .id_table =     USB4all_table,
};


static int __init usb_U4all_init(void)
{
	int result;

	/* register this driver with the USB subsystem */
	result = usb_register(&U4all_driver);
	if (result)
		err("usb_register failed. Error number %d", result);

	return result;
}

static void __exit usb_U4all_exit(void)
{
	/* deregister this driver with the USB subsystem */
	usb_deregister(&U4all_driver);
}

/******************************************************************************
 * unicode_to_ascii extracted from io_edgeport.c			      *		
 *	Turns a string from Unicode into ASCII.                               *
 *      Doesn't do a good job with any characters that are outside the normal *
 *      ASCII range, but it's only for debugging...                           *
 *      NOTE: expects the unicode in LE format				      *						
 *****************************************************************************/

static void unicode_to_ascii (char *string, __le16 *unicode, int unicode_size)
{
        int i;

        if (unicode_size <= 0)
                return;

        for (i = 0; i < unicode_size; ++i)
                string[i] = (char)(le16_to_cpu(unicode[i]));
        string[unicode_size] = 0x00;
}

module_init (usb_U4all_init);
module_exit (usb_U4all_exit);

MODULE_LICENSE("GPL");
